<?php

namespace App\Filament\Forms;

use App\Enums\BLTypes;
use App\Models\Company;
use Filament\Forms\Get;
use Filament\Forms\Set;
use App\Enums\CargoTypes;
use App\Classes\TzDateTime;
use App\Enums\LoadingTypes;
use App\Models\Organization;
use App\Enums\ContainerSizes;
use Filament\Forms\Components\Grid;
use App\Filament\Forms\CompanyForms;
use Filament\Forms\Components\Group;
use Filament\Forms\Components\Radio;
use Illuminate\Support\Facades\Auth;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\Wizard;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Fieldset;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Textarea;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Validation\Rules\Unique;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\RichEditor;
use Illuminate\Database\Eloquent\Builder;
use Filament\Forms\Components\ToggleButtons;
use Filament\Forms\Components\Actions\Action as FormsAction;

final class BLForms
{
    static public function reportGenForm() : Wizard {
        return Wizard::make([
            Wizard\Step::make('Search')
                ->schema([
                    Grid::make()
                        ->columns([
                            'sm' => 1,
                            'xl' => 2,
                        ])
                        ->schema([
                            Select::make('office_id')
                                ->label('Office')
                                ->relationship('office', 'name')
                                ->searchable(['name', 'code', 'website', 'primary_email', 'tel', 'mobile', 'post_code'])
                                ->getOptionLabelFromRecordUsing(fn (Model $record) => "{$record->name} ({$record->code}), {$record->country->name}")
                                ->visible(fn():bool => Auth::user()->isSuperAdmin() || Auth::user()->isAdmin())
                                ->columnSpanFull(),

                            Fieldset::make('Movement Info')
                                ->columns([
                                    'sm' => 1,
                                    'xl' => 2,
                                ])
                                ->schema([
                                    Select::make('place_of_loading_id')
                                        ->label('Place of Loading (POL)')
                                        ->relationship('place_of_loading', 'name')
                                        ->searchable(['name', 'iso', 'country_port_iso', 'country_name'])
                                        ->getOptionLabelFromRecordUsing(fn (Model $record) => "{$record->name} ({$record->iso}), {$record->country_name}")
                                        ->helperText(
                                            str('*POL (Manual)* will not work')->inlineMarkdown()->toHtmlString()
                                        ),
                                    Select::make('port_of_discharge_id')
                                        ->label('Place of Discharge (POD)')
                                        ->relationship('place_of_loading', 'name')
                                        ->searchable(['name', 'iso', 'country_port_iso', 'country_name'])
                                        ->getOptionLabelFromRecordUsing(fn (Model $record) => "{$record->name} ({$record->iso}), {$record->country_name}")
                                        ->helperText(
                                            str('*POD (Manual)* will not work')->inlineMarkdown()->toHtmlString()
                                        ),
                                    
                                    TextInput::make('place_of_receipt')
                                        ->label('Place of Receipt (POR)'),
                                    TextInput::make('final_destination')
                                        ->label('Final Destination Name'),
                                ]),

                            Fieldset::make('Voyage Info')
                                ->columns([
                                    'sm' => 1,
                                    'xl' => 3,
                                ])
                                ->schema([
                                    // Select::make('place_of_loading_id')
                                    //     ->label('POL (Place of Loading)')
                                    //     ->options(Port::all()->pluck('name', 'id'))
                                    //     ->default('ocean'),

                                    Select::make('vessel_type')
                                        ->options([
                                            'precarriage' => "Pre-Carriage Vessel",
                                            'ocean' => "Ocean Vessel",
                                        ])
                                        ->default('ocean'),
                                    
                                    TextInput::make('vessel')
                                        ->label('Vessel Name'),
                                    
                                    TextInput::make('voyage')
                                        ->label('Voyage Number'),
                                ]),
                            
                            // Fieldset::make('Est. Time of Departure')
                            //     ->columns([
                            //         'sm' => 1,
                            //         'xl' => 2,
                            //     ])
                            //     ->schema([
                            //         DatePicker::make('etd_from')
                            //             ->label('ETD From (date)'),
                            //         DatePicker::make('etd_until')
                            //             ->label('ETD Until (date)'),
                            //     ]),

                            Select::make('bl_type')
                                ->label('BL Type')
                                ->multiple()
                                ->options(BLTypes::class),

                            Select::make('bl_status')
                                ->label('BL Status')
                                ->multiple()
                                ->options(fn (): array => \App\Models\Status::query()->pluck('status', 'id')->all()),

                            Toggle::make('is_final')
                                ->label('Only Final BLs'),

                            Toggle::make('locked')
                                ->label('Only Locked BLs'),
                        ])
                ]),
            Wizard\Step::make('Time Period')
                ->schema([
                    Fieldset::make('Report Filter By')
                        ->columns(1)
                        ->schema([
                            ToggleButtons::make('filter_by')
                                ->label('Filter By Field')
                                ->options([
                                    'issue_date' => 'Issue Date',
                                    'eta' => 'ETA',
                                    'etd' => 'ETD',
                                    'created_at' => 'Submission Date',
                                ])
                                ->default('etd')
                                ->colors([
                                    'issue_date' => 'info',
                                    'eta' => 'warning',
                                    'etd' => 'success',
                                    'created_at' => 'lime',
                                ])
                                ->inline()
                                ->inlineLabel(false),
                        ]),
                    
                    ToggleButtons::make('report_type')
                        ->label('Generate Report By')
                        ->options([
                            'year' => 'Year',
                            'month' => 'Month',
                            // 'week' => 'Week',
                            'day' => 'Day',
                            'custom' => 'Custom',
                        ])
                        ->default('month')
                        ->colors([
                            'year' => 'info',
                            'month' => 'warning',
                            // 'week' => 'success',
                            'day' => 'agent',
                            'custom' => 'lime',
                        ])
                        ->inline()
                        ->inlineLabel(false)
                        ->grouped()
                        ->live(),

                    ToggleButtons::make('year')
                        ->label('Choose A Year')
                        ->options([
                            'this_year' => 'This Year ('. date('Y') .')',
                            'last_year' => 'Last Year ('. date('Y')-1 .')',
                            'last_two_years' => 'Last Two Years ('. date('Y')-1 .' - '. date('Y') .')',
                            'last_three_years' => 'Last Three Years ('. date('Y')-2 .' - '. date('Y') .')',
                            'last_six_years' => 'Last six Years ('. date('Y')-5 .' - '. date('Y') .')',
                            // 'last_twelve_years' => 'Last Twelve Years ('. date('Y')-11 .' - '. date('Y') .')',
                        ])
                        ->default('this_year')
                        ->colors([
                            'this_year' => 'info',
                            'last_year' => 'warning',
                            'last_two_years' => 'success',
                            'last_three_years' => 'agent',
                            'last_six_years' => 'lime',
                            'last_twelve_years' => 'sky',
                        ])
                        ->requiredIf('report_type', 'year')
                        ->inline()
                        ->inlineLabel(false)
                        // ->grouped()
                        ->visible(fn (Get $get): bool => $get('report_type')=='year'),
                        
                    ToggleButtons::make('month')
                        ->label('Choose A Month')
                        ->options([
                            'this_month' => 'This Month ('. date('M') .')',
                            'last_month' => 'Last Month ('. date('M', strtotime('-1 month')) .')',
                            'last_two_months' => 'Last Two Months ('. date('M', strtotime('-1 month')) .' - '. date('M') .')',
                            'last_three_months' => 'Last Three Months ('. date('M', strtotime('-2 months')) .' - '. date('M') .')',
                            'last_six_months' => 'Last six Months ('. date('M', strtotime('-5 months')) .' - '. date('M') .')',
                        ])
                        ->default('this_month')
                        ->colors([
                            'this_month' => 'info',
                            'last_month' => 'warning',
                            'last_two_months' => 'success',
                            'last_three_months' => 'agent',
                            'last_six_months' => 'lime',
                        ])
                        ->requiredIf('report_type', 'month')
                        ->inline()
                        ->inlineLabel(false)
                        // ->grouped()
                        ->visible(fn (Get $get): bool => $get('report_type')=='month'),
                    ToggleButtons::make('day')
                        ->label('Choose A Day')
                        ->options([
                            'today' => 'Today',
                            'yesterday' => 'Yesterday',
                        ])
                        ->default('today')
                        ->colors([
                            'today' => 'info',
                            'yesterday' => 'warning',
                        ])
                        ->requiredIf('report_type', 'day')
                        ->inline()
                        ->inlineLabel(false)
                        // ->grouped()
                        ->visible(fn (Get $get): bool => $get('report_type')=='day'),

                    Fieldset::make('Custom Date Ranage')
                        ->columns([
                            'sm' => 1,
                            'xl' => 2,
                        ])
                        ->schema([
                            DatePicker::make('date_from')
                                ->label('From (date)')
                                ->default('-7days')
                                ->requiredIf('report_type', 'custom'),
                            DatePicker::make('date_until')
                                ->label('Until (date)')
                                ->default('today')
                                ->requiredIf('report_type', 'custom'),
                        ])
                        ->visible(fn (Get $get): bool => $get('report_type')=='custom'),
                ]),
            // Wizard\Step::make('Columns')
            //     ->schema([
            //         ToggleButtons::make('columns')
            //             ->hint('Selected columns will show on the report!')
            //             ->label('Choose Extra Columns')
            //             ->multiple()
            //             ->options([
            //                 'created_at' => 'Submission Date',
            //                 'updated_at' => 'Last Updated',
            //                 'posted_by_user' => 'Submitted By',
            //             ])
            //             ->colors([
            //                 'created_at' => 'info',
            //                 'updated_at' => 'warning',
            //                 'posted_by_user' => 'success',
            //             ])
            //             ->inline()
            //             ->inlineLabel(false)
            //             // ->grouped(),
            //     ]),
        ]);
    }

    static public function makeBlLeftForm(bool $customer = false, array $options = []): Group {
        // left side
        return Group::make()
        ->schema([
            self::makeBlGeneralSection(),
            self::makeBlVesselSection(),
            self::makeBlDestinationSection(),
            self::makeBlContainerInfoSection(),
            self::makeBlContainerSection(),
            self::makeBlCertificateSection(),
            self::makeBlFrieghtSection(),
        ])
        ->columnSpan(['lg' => 2])
        ->disabled(fn (?Model $record): bool => !empty($record->locked));
    }

    static public function makeBlGeneralSection(string $description =''): Section {
        return Section::make('Voyage')
            // ->description('Description')
            ->schema([
                // TextInput::make('bl_no')->label('Bill Of Leading Number')
                //     ->helperText('Bill Of Leading (BL) number will generate automatically!')
                //     ->required()
                //     ->unique(ignoreRecord: true, modifyRuleUsing: function (Unique $rule) {
                //         return $rule->where('deleted_at', NULL)->whereNot('bl_no', 'draft');
                //     })
                //     ->disabled()
                //     ->columnSpan(2),
                DatePicker::make('etd')
                    ->label('Estimated time of departure')
                    ->required()
                    ->default('+1days'),
                DatePicker::make('eta')
                    ->label('Estimated time of arrival')
                    ->required()
                    ->default('+7days'),
                TextInput::make('pre_carriage_vessel')
                    ->label('Pre-carriage Vessel'),
                TextInput::make('voy_no_pre_carriage')
                    ->label('Voyage No (Pre-carriage)'),
                Select::make('ocean_vessel_id')
                    ->relationship('ocean_vessel', 'name')
                    ->searchable(['name', 'code'])
                    ->preload()
                    // ->required(fn(Callable $get):bool => empty($get('ocean_vessel_old')))
                    ->requiredWithout('ocean_vessel_old')
                    ->label('Ocean Vessel'),
                Select::make('ocean_vessel_voy_id')
                    ->relationship(
                        name: 'ocean_vessel_voy',
                        titleAttribute: 'voy_no',
                        modifyQueryUsing: fn (Builder $query) => $query->whereHas('transits', function(Builder $query){
                            $query->where('eta', '>=', date('Y-m-d'));
                        }))
                    ->searchable()
                    ->preload()
                    // ->required(fn(Callable $get):bool => empty($get('voy_no_destination_old')))
                    ->requiredWithout('voy_no_destination_old')
                    ->label('Ocean Vessel Voyage'),
                TextInput::make('ocean_vessel_old')
                    // ->required(fn(Callable $get):bool => empty($get('ocean_vessel_id')))
                    ->requiredWithout('ocean_vessel_id')
                    // ->readOnly()
                    ->helperText('This field is will be removed soon, select ocean vessel instead')
                    // ->hidden(fn(string|null $state):bool=>empty($state))
                    ->label('Ocean Vessel'),
                TextInput::make('voy_no_destination_old')
                    // ->required(fn(Callable $get):bool => empty($get('ocean_vessel_voy_id')))
                    ->requiredWithout('ocean_vessel_voy_id')
                    // ->readOnly()
                    ->helperText('This field is will be removed soon, select ocean vessel voyage instead')
                    // ->hidden(fn(string|null $state):bool=>empty($state))
                    ->label('Voyage No (Destination)'),
            ])->columns(2);
    }

    static public function makeBlVesselSection(string $description = ''): Section {
        return Section::make('Locations')
            ->description('Add primary location for your shipment!')
            ->schema([
                // Discharge / Destination
                Select::make('place_of_loading_id')
                    ->label('Place of Loading (POL)')
                    ->relationship('place_of_loading', 'name')
                    ->searchable(['name', 'iso', 'country_port_iso', 'country_name'])
                    ->getOptionLabelFromRecordUsing(fn (Model $record) => "{$record->name} ({$record->iso}), {$record->country_name}")
                    ->different('port_of_discharge_id')
                    ->reactive()
                    ->required()
                    ->helperText(
                        str('Select a Port or Write your value at below')->inlineMarkdown()->toHtmlString()
                    )
                    ->columnSpan(3),
                Select::make('port_of_discharge_id')
                    ->label('Port of Discharge (POD)')
                    ->relationship('port_of_discharge', 'name')
                    ->searchable(['name', 'iso', 'country_port_iso', 'country_name'])
                    ->getOptionLabelFromRecordUsing(fn (Model $record) => "{$record->name} ({$record->iso}), {$record->country_name}")
                    ->different('place_of_loading_id')
                    ->reactive()
                    ->required()
                    ->helperText(
                        str('Select a Port or Write your value at below')->inlineMarkdown()->toHtmlString()
                    )
                    ->columnSpan(3),
                TextInput::make('place_of_loading_manual')
                    ->label('Place of Loading (Manual)')
                    ->helperText(
                        str('Custom text for **Place Of Loading** on BL.')->inlineMarkdown()->toHtmlString()
                    )
                    ->columnSpan(3),
                TextInput::make('port_of_discharge_manual')
                    ->label('Port of Discharge (Manual)')
                    ->helperText(
                        str('Custom text for **Port of Discharge** on BL.')->inlineMarkdown()->toHtmlString()
                    )
                    ->columnSpan(3),
                TextInput::make('place_of_receipt')
                    ->label('Place of Receipt (POR)')
                    ->required()
                    ->columnSpan(6),
            ])->columns(6);
    }

    static public function makeBlDestinationSection(string $description = ''): Section {
        // Destination 
        return Section::make('Destination')
            ->description('We will connect our agent based on your destination info')
            ->schema([
                TextInput::make('final_destination')
                    ->label('Final Destination Name')
                    ->required()
                    ->columnSpan(2),
                Select::make('final_destination_country_id')
                    ->label('Final Destination Country')
                    // ->relationship('final_destination_country', 'name')
                    ->options(\App\Models\Country::all()->pluck('name', 'id'))
                    ->reactive()
                    ->afterStateUpdated(function (callable $set) {
                        $set('final_destination_city_id', '');
                    })
                    ->searchable(['name', 'iso3', 'iso2', 'numeric_code', 'capital', 'currency'])
                    ->exists(table: \App\Models\Office::class, column: 'country_id')
                    ->validationMessages([
                        'exists' => 'Sorry! We don\'t have any office there.',
                    ]),
                Select::make('final_destination_city_id')
                    ->label('Final Destination City')
                    // ->relationship('final_destination_city', 'name')
                    ->options(function(callable $get) {
                        $country = \App\Models\Country::find($get('final_destination_country_id'));
                        if ($country) {
                            return $country->cities->pluck('name', 'id');
                        }
                        return [''=>'Please Choose A Country'];
                    })
                    // ->required(fn (Get $get): bool => filled($get('company_name')))
                    ->requiredWith('final_destination_country_id')
                    ->searchable(['name', 'country_code'])
                    ->exists(table: \App\Models\Office::class, column: 'city_id')
                    ->validationMessages([
                        'exists' => 'Sorry! We don\'t have any office there.',
                    ]),
                
                Textarea::make('delivery_agent_manual')
                    ->label('Delivery Agent (Manual) - Name & Address')
                    ->helperText(str('Hit **⏎ Enter** to add new line!')->inlineMarkdown()->toHtmlString())
                    ->requiredWithoutAll('final_destination_country_id,final_destination_city_id')
                    ->columnSpan(2),
            ])->columns(2);
    }

    static public function makeBlContainerInfoSection(string $description = ''): Section {
        // Container Info 
        return Section::make('Container Info')
            // ->description('Description')
            ->schema([
                TextInput::make('weight')
                    ->label('Gross Weight')
                    ->helperText('Calculate Automatically')
                    ->disabled()
                    ->postfix('KGs')
                    ->default('0')
                    ->live(),
                TextInput::make('net_weight')
                    ->disabled()
                    ->helperText('Calculate Automatically')
                    ->postfix('KGs')
                    ->default('0')
                    ->live()
                    ->label('Net Weight'),
                TextInput::make('measurement')
                    ->disabled()
                    ->helperText('Calculate Automatically')
                    ->postfix('m3')
                    ->default('0')
                    ->live(),
                TextInput::make('qty')
                    ->disabled()
                    ->helperText('Calculate Automatically')
                    ->postfix('Pkgs')
                    ->live()
                    ->default('0')
                    // ->required()
                    ->label('No. Of Packages'),
                Textarea::make('discription_of_goods')
                    ->rows(5)
                    ->required()
                    ->maxLength(500)
                    ->columnSpan(4)
                    ->hint('Maximum 500 characters')
                    ->hintIcon('heroicon-o-information-circle'),
                Textarea::make('marks')
                    ->rows(5)
                    ->maxLength(500)
                    ->columnSpan(4)
                    ->hint('Maximum 500 characters')
                    ->hintIcon('heroicon-o-information-circle'),
                Textarea::make('rider_sheet')->label('Rider Sheet (If required)')
                    ->rows(5)
                    ->autosize()
                    ->columnSpan(4),
            ])->columns(4);
    }
    
    static public function makeBlContainerSection(string $description = ''): Repeater {
        return Repeater::make('bl_containers')
            ->hint('Duplicate Container & Seal number are not allowed.')
            ->hintIcon('heroicon-o-information-circle')
            ->label('Containers')
            ->schema([
                TextInput::make('container_num')
                    ->label('Container Number')
                    ->required()
                    ->distinct()
                    ->live()
                    ->helperText('Unique Number Only')
                    ->columnSpan(6),
                TextInput::make('seal_no')
                    ->label('Seal number')
                    ->helperText('Unique Number Only')
                    ->distinct()
                    ->live()
                    ->required()
                    ->columnSpan(6),

                Select::make('container_size')
                    ->options(ContainerSizes::class)
                    ->default('20_hc')
                    ->required()
                    ->searchable()
                    ->columnSpan(4),
                TextInput::make('pack_qty')
                    ->label('Qty (pcs/pck)')
                    ->columnSpan(4),
                TextInput::make('measurement')
                    ->label('Measurement')
                    ->numeric()
                    ->minLength(1)
                    ->extraInputAttributes([
                        'min' => 1
                    ])
                    ->postfix('m3')
                    // ->live()
                    ->columnSpan(4),

                TextInput::make('gross_weight')
                    ->label('Gross Weight (KGs)')
                    ->numeric()
                    ->minLength(1)
                    ->extraInputAttributes([
                        'min' => 1
                    ])
                    ->postfix('KGs')
                    // ->live()
                    // ->afterStateUpdated(function (Set $set, Get $get) {
                    //     self::container_info($get, $set);
                    //     // dd($get('../*'));
                    //     // // dd((int)$get('../../weight'));
                    //     // $weight = (int)$get('../../weight');
                    //     // // dd($weight);
                    //     // return $set('../../weight', (0+$state));
                    // })
                    ->columnSpan(4),
                TextInput::make('net_weight')
                    ->label('Net Weight (KGs)')
                    ->numeric()
                    ->minLength(1)
                    ->extraInputAttributes([
                        'min' => 1
                    ])
                    ->postfix('KGs')
                    // ->live()
                    ->columnSpan(4),
                TextInput::make('vgm')
                    ->label('VGM (KGs)')
                    ->columnSpan(4),
            ])
            ->relationship()
            ->cloneable()
            ->collapsible()
            // ->orderable()
            ->minItems(1)
            ->columns(12)
            ->live(onBlur: true)
            ->afterStateUpdated(function (Get $get, Set $set) {
                self::container_info($get, $set);
            })
            ->deleteAction(
                fn(FormsAction $action) => $action->after(fn(Get $get, Set $set) => self::container_info($get, $set)),
            )
            // ->itemLabel(fn (array $state): ?string => ($state['container_num'] . ' (' . Str::upper(Str::replace('_', '\' ', $state['container_size'])) . ')' . ' / ' . $state['seal_no']) ?? null)
            ->itemLabel(fn (array $state): ?string => !empty($state['container_num'])?($state['container_num'] . ' (' . ContainerSizes::tryFrom($state['container_size'])->getLabel() . ')' . ' / ' . $state['seal_no']) : null)
            ->addActionLabel('Add More Container');
    }

    static public function makeBlCertificateSection(string $description = ''): Repeater {
        return Repeater::make('bl_certificates')
            ->label('Certificates')
            ->schema([
                TextInput::make('cer_title')
                    ->label('Certificate Title')
                    ->required(),
                RichEditor::make('certificate')
                    ->label('Certificate Content')
                    ->disableToolbarButtons([
                        'attachFiles',
                    ])
                    ->required(),
            ])
            ->relationship()
            ->cloneable()
            ->collapsible()
            ->itemLabel(fn (array $state): ?string => $state['cer_title'] ?? null)
            ->addActionLabel('Add Certificate')
            ->visible(fn():bool => !auth()->user()->is_customer());
    }

    static public function makeBlFrieghtSection(string $description = ''): Section {
        // Frieght Info
        return Section::make('Frieght Info')
            // ->description('Description')
            ->schema([
                Select::make('freight_type')
                    ->options([
                        'prepaid' => 'Prepaid', 
                        'postpaid' => 'Postpaid', 
                        'collect' => 'Collect'
                        ])
                    ->default('prepaid')
                    ->required()
                    ->columnSpan(3),
                Textarea::make('freight_and_charges')
                    // ->required()
                    ->columnSpan(3),
                TextInput::make('revenue_tons'),
                TextInput::make('rate')
                    ->numeric(),
                TextInput::make('prepaid'),
                TextInput::make('collect'),
                TextInput::make('ex_rate')
                    ->label('Exchange Rate (USD)')
                    ->numeric(),
                TextInput::make('prepaid2')
                    ->required()
                    ->label('Prepaid (Upto Port)'),
            ])->columns(3);
    }

    static public function makeBlRightForm(bool $customer = false, array $options = []): Group {
        // right side
        return Group::make()
        ->schema([
            self::makeBlStatusSection(),
            self::makeBlCustomerSection(),
            self::makeBlReferenceSection(),
            self::makeBlPaymentInfoSection(),
        ])
        ->columnSpan(['lg' => 1]);
    }

    static public function makeBlStatusSection(string $description = ''): Section {
        return Section::make('Status')
            // ->description('Description')
            ->schema([
                Grid::make([
                    'default' => 1,
                    'sm' => 2,
                ])
                ->schema([
                    Toggle::make('is_final')->label('Final'),
                    Toggle::make('locked'),
                ])
                ->visible(fn():bool => !auth()->user()->is_customer()),
                Select::make('editing_time')
                    ->label('Editing Time Allowed')
                    // ->live()
                    // ->reactive()
                    ->default(1*24*60*60)
                    ->helperText(fn(?Model $record):string|null => !empty($record)?'Created ' . $record->created_at->timezone(TzDateTime::getOfficeTimeZone()->zoneName)->format('d-M-Y h:i A'):null)
                    // ->helperText(fn(?Model $record):string|null => !empty($record)?dd(TzDateTime::getOfficeTimeZone()->zoneName):null)
                    ->options([
                        1*24*60*60 => '1 Day',
                        2*24*60*60 => '2 Days',
                        3*24*60*60 => '3 Days',
                        4*24*60*60 => '4 Days',
                        5*24*60*60 => '5 Days',
                        6*24*60*60 => '6 Days',
                        7*24*60*60 => '7 Days',
                        8*24*60*60 => '8 Days',
                        9*24*60*60 => '9 Days',
                        10*24*60*60 => '10 Days',
                        11*24*60*60 => '11 Days',
                        12*24*60*60 => '12 Days',
                        13*24*60*60 => '13 Days',
                        14*24*60*60 => '14 Days',
                    ])
                    ->visible(fn():bool => !auth()->user()->is_customer()),
                Select::make('bl_type')
                    ->label('BL Type')
                    ->options(BLTypes::class)
                    ->default('draft')
                    ->required(),
                Select::make('bl_status')
                    ->label('BL Status')
                    ->options(\App\Models\Status::all()->pluck('status', 'id'))
                    ->required()
                    ->default(1),
                
                Select::make('cargo_type')
                    ->options(CargoTypes::class)
                    ->searchable()
                    // ->columnSpan(4)
                    ->required(),
                // Select::make('loading_type')
                //     ->options(LoadingTypes::class)
                //     ->default('cy_cy')
                //     ->columnSpan(2)
                //     ->required()
                //     ->hidden(),
                // RichEditor::make('container_info')->columnSpan(4),
                TextInput::make('commodity')
                    ->helperText('It will only show in Tracking Result page')
                    ->visible(fn():bool => !auth()->user()->is_customer()),
            ]);
    }

    static public function makeBlCustomerSection(string $description = ''): Section {
        return Section::make('Customer')
            // ->description('First line is considered as Company Name and other lines as Address')
            ->description('Write Name on first line and Address below of it!')
            ->schema([
                Select::make('organization_id')
                    ->label('Organization / Customer')
                    // ->options(dd(\App\Models\Organization::all()->pluck('name', 'id')))
                    // ->options(\App\Models\Organization::withoutGlobalScopes(['office'])->pluck('name', 'id'))
                    ->relationship(
                            name: 'organization', 
                            titleAttribute: 'name',
                            // modifyQueryUsing: fn (Builder $query) => $query->withoutGlobalScopes(['office'])
                        )
                    ->getOptionLabelFromRecordUsing(fn (Model $record) => "{$record->name} ({$record->code})")
                    ->preload()
                    ->reactive()
                    ->helperText(function(Callable $get) {
                        $company = Organization::find($get('organization_id'));
                        $address = $company?->address;
                        return !empty($company)?'Address: '. $address:'Select an organization';
                    })
                    ->default(request('organization_id'))
                    ->searchable(['name', 'code', 'website', 'email'])
                    ->visible(fn():bool => !auth()->user()->is_customer()),
                Select::make('shipper_company_id')
                    ->label('Shipper (Select Company)')
                    ->relationship(
                        name: 'shipper', 
                        titleAttribute: 'name', 
                        modifyQueryUsing: function (Builder $query, Callable $get) {
                            // if ( empty($get('shipper_company_id')) && !empty($get('organization_id')) ) {
                            //     return $query->where('organization_id', $get('organization_id'))->withoutGlobalScopes();
                            // }
                            return $query;
                        })
                    ->getOptionLabelFromRecordUsing(fn (Model $record) => "{$record->name} ({$record->code})")
                    ->searchable(['name', 'code', 'website', 'primary_email'])
                    ->searchPrompt('Search by name, code, website or email')
                    ->optionsLimit(10)
                    ->required(fn(?Model $record):bool => empty($record->shipper_text))
                    ->reactive()
                    ->helperText(function(Callable $get) {
                        $company = Company::find($get('shipper_company_id'));
                        $address = $company?->address . ', ' . $company?->city->name . ', ' . $company?->country->iso2;
                        return !empty($company)?'Address: '. $address:'Select a company';
                    })
                    ->createOptionForm([
                        CompanyForms::getAddForm()
                    ]),
                Textarea::make('shipper_text')
                    ->label('Shipper Name & Address')
                    ->required(fn(?Model $record):bool => !empty($record->shipper_text))
                    ->visible(fn(?Model $record):bool => !empty($record->shipper_text))
                    ->disabled(fn(?Model $record):bool => !empty($record->shipper_text))
                    ->helperText(str('Hit **⏎ Enter** to add new line!')->inlineMarkdown()->toHtmlString()),

                Select::make('consignee_company_id')
                    ->label('Consignee (Select Company)')
                    ->relationship(
                        name: 'consignee', 
                        titleAttribute: 'name', 
                        modifyQueryUsing: function (Builder $query, Callable $get) {
                            // if ( empty($get('consignee_company_id')) && !empty($get('organization_id')) ) {
                            //     return $query->where('organization_id', $get('organization_id'))->withoutGlobalScopes();
                            // }
                            return $query;
                        })
                    ->getOptionLabelFromRecordUsing(fn (Model $record) => "{$record->name} ({$record->code})")
                    ->searchable(['name', 'code', 'website', 'primary_email'])
                    ->optionsLimit(10)
                    ->required(fn(?Model $record):bool => empty($record->shipper_text))
                    ->reactive()
                    ->helperText(function(Callable $get) {
                        $company = Company::find($get('consignee_company_id'));
                        $address = $company?->address . ', ' . $company?->city->name . ', ' . $company?->country->iso2;
                        return !empty($company)?'Address: '. $address:'Select a company';
                    })
                    ->createOptionForm([
                        CompanyForms::getAddForm()
                    ]),
                Textarea::make('consignee_text')
                    ->label('Consignee Name & Address')
                    ->required(fn(?Model $record):bool => !empty($record->consignee_text))
                    ->visible(fn(?Model $record):bool => !empty($record->consignee_text))
                    ->disabled(fn(?Model $record):bool => !empty($record->consignee_text))
                    ->helperText(str('Hit **⏎ Enter** to add new line!')->inlineMarkdown()->toHtmlString()),

                Select::make('notify_party_company_id')
                    ->label('1st Notify Party (Select Company)')
                    ->relationship(
                        name: 'notify_party', 
                        titleAttribute: 'name', 
                        modifyQueryUsing: function (Builder $query, Callable $get) {
                            // if ( empty($get('notify_party_company_id')) && !empty($get('organization_id')) ) {
                            //     return $query->where('organization_id', $get('organization_id'))->withoutGlobalScopes();
                            // }
                            return $query;
                        })
                    ->getOptionLabelFromRecordUsing(fn (Model $record) => "{$record->name} ({$record->code})")
                    ->searchable(['name', 'code', 'website', 'primary_email'])
                    ->optionsLimit(10)
                    ->required(fn(?Model $record):bool => empty($record->shipper_text))
                    ->reactive()
                    ->helperText(function(Callable $get) {
                        $company = Company::find($get('notify_party_company_id'));
                        $address = $company?->address . ', ' . $company?->city->name . ', ' . $company?->country->iso2;
                        return !empty($company)?'Address: '. $address:'Select a company';
                    })
                    ->createOptionForm([
                        CompanyForms::getAddForm()
                    ]),
                Textarea::make('notify_party_text')
                    ->label('1st Notify Party Name & Address')
                    ->required(fn(?Model $record):bool => !empty($record->notify_party_text))
                    ->visible(fn(?Model $record):bool => !empty($record->notify_party_text))
                    ->disabled(fn(?Model $record):bool => !empty($record->notify_party_text))
                    ->helperText(str('Hit **⏎ Enter** to add new line!')->inlineMarkdown()->toHtmlString()),

                Select::make('notify_party2_company_id')
                    ->label('2nd Notify Party (Select Company)')
                    ->relationship(
                        name: 'notify_party2', 
                        titleAttribute: 'name', 
                        modifyQueryUsing: function (Builder $query, Callable $get) {
                            // if ( empty($get('notify_party2_company_id')) && !empty($get('organization_id')) ) {
                            //     return $query->where('organization_id', $get('organization_id'))->withoutGlobalScopes();
                            // }
                            return $query;
                        })
                    ->getOptionLabelFromRecordUsing(fn (Model $record) => "{$record->name} ({$record->code})")
                    ->searchable(['name', 'code', 'website', 'primary_email'])
                    ->optionsLimit(10)
                    ->reactive()
                    ->helperText(function(Callable $get) {
                        $company = Company::find($get('notify_party2_company_id'));
                        $address = $company?->address . ', ' . $company?->city->name . ', ' . $company?->country->iso2;
                        return !empty($company)?'Address: '. $address:'Select a company';
                    })
                    ->createOptionForm([
                        CompanyForms::getAddForm()
                    ]),
                Textarea::make('notify_party2_text')
                    ->visible(fn(?Model $record):bool => !empty($record->notify_party2_text))
                    ->disabled(fn(?Model $record):bool => !empty($record->notify_party2_text))
                    ->label('2nd Notify Party Name & Address')
                    ->helperText(str('Hit **⏎ Enter** to add new line!')->inlineMarkdown()->toHtmlString()),
            ])
            ->disabled(fn (?Model $record): bool => !empty($record->locked));
    }

    static public function makeBlReferenceSection(string $description = ''): Section {
        return Section::make('Reference')
            // ->description('Description')
            ->schema([
                TextInput::make('booking_ref_no')
                    ->label('Booking reference number'),
                TextInput::make('shipper_ref')
                    ->label('Shipper reference number'),
                Radio::make('original_bl')
                    ->label('Number of Original BL(s)')
                    ->options([
                        1 => "1 (One)", 
                        2 => "2 (Two)", 
                        3 => "3 (Three)"
                    ])
                    ->inline()
                    ->inlineLabel(false)
                    ->default(1)
                    ->required(),
                Select::make('original_text')
                    ->label('BL Original Text')
                    ->options([
                        '1st Original' => '1st Original',
                        '2nd Original' => '2nd Original',
                        '3rd Original' => '3rd Original',
                        'Non-negotiable' => 'Non-negotiable'
                    ])
                    ->visible(fn():bool => !auth()->user()->is_customer()),
                DatePicker::make('shipped_on_board'),
                    // ->required()
                    // ->default('+1days')
                    // ->helperText(str('Leave empty if date is same as **ETA**!')->inlineMarkdown()->toHtmlString()),
                DatePicker::make('rcv_for_shipment')
                    ->label('Received for Shipment'),
                    // ->required()
                    // ->default('+1days')
                    // ->helperText(str('Leave empty if date is same as **ETA**!')->inlineMarkdown()->toHtmlString()),
            ])
            ->disabled(fn (?Model $record): bool => !empty($record->locked));
    }

    static public function makeBlPaymentInfoSection(string $description = ''): Section {
        // Payment Info
        return Section::make('Payment Info')
            // ->description('Description')
            ->schema([
                Select::make('invoice_to_company_id')
                    ->label('Invoice To (Company)')
                    // ->options(\App\Models\Port::all()->pluck('name', 'id'))
                    ->relationship('invoice_to_company', 'name')
                    ->getOptionLabelFromRecordUsing(fn (Model $record) => "{$record->name} ({$record->code})")
                    ->searchable(['name', 'code', 'website', 'primary_email'])
                    ->reactive()
                    ->createOptionForm([
                        CompanyForms::getAddForm()
                    ]),
                TextInput::make('payable_at')
                    ->required(),
                TextInput::make('place_issue')
                    ->required()
                    ->label('Place Of Issue'),
                DatePicker::make('issue_date')
                    ->helperText(str('Leave empty if date is same as **ETA**!')
                    // ->default(today())
                    ->inlineMarkdown()->toHtmlString()),
            ])
            ->disabled(fn (?Model $record): bool => !empty($record->locked));
    }

    static public function container_info(Get $get, Set $set) : void {
        $total_gross_weight = 0;
        $total_net_weight = 0;
        $total_measurement = 0;
        $total_pkg = 0;
        $pkg_type = '';

        // dd($get('bl_containers'), $get('*'), $get('../*'));

        foreach ($get('bl_containers') as $item) {
            $pkgs = [preg_replace('/[^0-9]/', '', $item['pack_qty']), preg_replace('/[^a-zA-Z]/', '', $item['pack_qty'])];
            // dd($pkgs, $item['pack_qty']);

            if (empty($pkg_type)) {
                $pkg_type = $pkgs[1];
            } elseif (strtolower($pkg_type) != strtolower($pkgs[1])) {
                $pkg_type = 'Packages';
            }

            $total_gross_weight += (int) $item['gross_weight'];
            $total_net_weight += (int) $item['net_weight'];
            $total_measurement += (int) $item['measurement'];
            $total_pkg += (int) $pkgs[0];
        }
        // dd($get('weight'), $get('net_weight'), $get('measurement'));
        // dd($total_gross_weight, $total_net_weight, $total_measurement);

        $set('weight', $total_gross_weight);
        $set('net_weight', $total_net_weight);
        $set('measurement', $total_measurement);
        $set('qty', ($total_pkg.' '.ucwords($pkg_type)));
    }
}