<?php

namespace App\Filament\Office\Resources;

use App\Models\BL;
use Filament\Tables;
use App\Enums\BLTypes;
use Filament\Forms\Form;
use Filament\Tables\Table;
use App\Filament\Forms\BLForms;
use Filament\Resources\Resource;
use App\Filament\Tables\BLTables;
use Filament\Tables\Actions\Action;
use Filament\Tables\Filters\Filter;
use Illuminate\Support\Facades\Auth;
use Illuminate\Database\Eloquent\Model;
use Filament\Notifications\Notification;
use Filament\Forms\Components\DatePicker;
use Filament\Tables\Filters\SelectFilter;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Enums\ActionsPosition;
use Filament\Tables\Actions\ReplicateAction;
use App\Filament\Office\Resources\BLResource\Pages;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use App\Filament\Office\Resources\BLResource\RelationManagers;

class BLResource extends Resource
{
    protected static ?string $model = BL::class;

    protected static ?string $slug = 'shipping/bls';

    protected static ?string $navigationGroup = 'Shipping';
    protected static ?string $navigationIcon = 'heroicon-o-ticket';
    protected static ?string $navigationLabel = 'Bill Of Ladings';
    protected static ?string $modelLabel = 'Bill Of Lading';
    
    public static function shouldRegisterNavigation(): bool
    {
        // dd(auth()->user()->can('viewAny', BL::class));
        // return auth()->user() ? auth()->user()->has('office')->count() : false;
        return auth()->user()?auth()->user()->can('viewAny', BL::class):false;
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                BLForms::makeBlLeftForm(customer: false),
                BLForms::makeBlRightForm(customer: false),
            ])
            ->columns(3);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns(
                BLTables::blTable(customer: false),
            )
            ->defaultSort('created_at', 'desc')
            // ->recordClasses(function(Model $record){
            //     $class_opacity = match ($record->locked) {
            //         1 => 'opacity-30',
            //         default => null,
            //     };
            //     $class_final = match ($record->is_final) {
            //         1 => 'border-s-2 border-agent-600 dark:border-agent-300',
            //         default => null,
            //     };
            //     // 'reviewing' => 'border-s-2 border-orange-600 dark:border-orange-300',
            //     // 'published' => 'border-s-2 border-green-600 dark:border-green-300',
            //     return $class_opacity .' '. $class_final;
            // })
            // ->striped()
            ->searchOnBlur()
            ->filters([
                SelectFilter::make('organization')
                    // ->relationship('organization', 'name', fn (Builder $query) => $query->withoutGlobalScopes(['office']))
                    ->relationship('organization', 'name')
                    ->multiple()
                    ->preload()
                    ->attribute('organization_id')
                    ->searchable(),
                
                SelectFilter::make('office')
                    // ->relationship('office', 'name', fn (Builder $query) => $query->withoutGlobalScopes(['office']))
                    ->relationship('office', 'name')
                    ->multiple()
                    ->preload()
                    ->attribute('office_id')
                    ->searchable()
                    ->visible(fn():bool => Auth::user()->isSuperAdmin() || Auth::user()->isAdmin()),

                Filter::make('etd')
                    ->form([
                        DatePicker::make('etd_from')
                            ->label('ETD From (date)'),
                        DatePicker::make('etd_until')
                            ->label('ETD Until (date)'),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when(
                                $data['etd_from'],
                                fn (Builder $query, $date): Builder => $query->whereDate('etd', '>=', $date),
                            )
                            ->when(
                                $data['etd_until'],
                                fn (Builder $query, $date): Builder => $query->whereDate('etd', '<=', $date),
                            );
                    }),
                
                // Filter::make('eta')
                //     ->form([
                //         DatePicker::make('eta_from')
                //             ->label('ETA From (date)'),
                //         DatePicker::make('eta_until')
                //             ->label('ETA Until (date)'),
                //     ])
                //     ->query(function (Builder $query, array $data): Builder {
                //         return $query
                //             ->when(
                //                 $data['eta_from'],
                //                 fn (Builder $query, $date): Builder => $query->whereDate('eta', '>=', $date),
                //             )
                //             ->when(
                //                 $data['eta_until'],
                //                 fn (Builder $query, $date): Builder => $query->whereDate('eta', '<=', $date),
                //             );
                //     }),
                Tables\Filters\SelectFilter::make('bl_type')
                    ->multiple()
                    ->options(BLTypes::class),
                Tables\Filters\SelectFilter::make('bl_status')
                    ->multiple()
                    ->options(fn (): array => \App\Models\Status::query()->pluck('status', 'id')->all()),
                Filter::make('is_final')
                    ->label('Only Final BLs')
                    ->toggle()
                    ->query(fn (Builder $query): Builder => $query->where('is_final', true)),
                Filter::make('locked')
                    ->label('Only Locked BLs')
                    ->toggle()
                    ->query(fn (Builder $query): Builder => $query->where('locked', true)),
                Tables\Filters\TrashedFilter::make(),
            ])
            ->deferFilters()
            ->actions([
                Tables\Actions\ActionGroup::make([
                    Action::make('print_ridersheet')
                        ->label('Rider Sheet')
                        ->url(fn (?Model $record): string => route('shipping.bl.ridersheet', $record->bl))
                        ->icon('heroicon-o-newspaper')
                        // ->hidden(fn (?Model $record): string => $record->hasRiderSheet())
                        ->visible(fn (?Model $record): bool => $record->hasRiderSheet())
                        // ->color('lime')
                        ->openUrlInNewTab(),

                    Action::make('print_certificate')
                        ->label('Certificate')
                        ->url(fn (?Model $record): string => route('shipping.bl.certificate', $record->bl))
                        ->icon('heroicon-o-identification')
                        ->hidden(fn (?Model $record): string => $record->bl_certificates->count() < 1)
                        // ->color('lime')
                        ->openUrlInNewTab(),

                    Action::make('print')
                        ->url(fn (?Model $record): string => route('shipping.bl.show', $record->bl))
                        ->icon('heroicon-o-printer')
                        ->openUrlInNewTab(),
                    
                    Action::make('pdf')
                        ->label('PDF')
                        ->url(fn (?Model $record): string => route('shipping.bl.pdf', $record->bl))
                        ->icon('heroicon-o-document')
                        // ->color('success')
                        ->openUrlInNewTab(),

                    ReplicateAction::make()
                        ->excludeAttributes(['notify_party2', 'qty', 'discription_of_goods', 'marks', 'rider_sheet', 'weight', 'net_weight', 'measurement', 'place_issue', 'issue_date', 'shipped_on_board', 'issue_date', 'created_at','customer_created', 'bl_containers_count'])
                        // ->mutateRecordDataUsing(function (array $data): array {
                        //     $data['shipper'] = 'draft';
                        //     $data['consignee'] = 'draft';
                        //     $data['notify_party'] = 'draft';
                        //     $data['bl_no'] = 'draft';
                        //     $data['posted_user_type'] = 2;
                        //     $data['posted_by_user_id'] = auth()->id();
                     
                        //     return $data;
                        // })
                        ->beforeReplicaSaved(function (Model $replica): Model {
                            $replica->shipper = 'draft';
                            $replica->consignee = 'draft';
                            $replica->notify_party = 'draft';
                            $replica->bl_no = 'draft';
                            $replica->posted_user_type = 2;
                            $replica->posted_by_user_id = auth()->id();

                            return $replica;
                        })
                        ->successRedirectUrl(fn (Model $replica): string => route('filament.office.resources.shipping.bls.edit', [
                            'record' => $replica,
                        ]))
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title('BL replicated')
                                ->body('The BL has been replicated successfully.'),
                        ),
                    Action::make('activities')->url(fn ($record) => BLResource::getUrl('activities', ['record' => $record]))->icon('heroicon-o-rectangle-stack'),
                    Tables\Actions\EditAction::make()->icon('heroicon-o-pencil-square'),
                    Tables\Actions\DeleteAction::make()->icon('heroicon-o-trash'),
                ]),
            ], position: ActionsPosition::BeforeColumns)
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                    Tables\Actions\RestoreBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListBLS::route('/'),
            'create' => Pages\CreateBL::route('/create'),
            'edit' => Pages\EditBL::route('/{record}/edit'),
            'activities' => Pages\ListBLActivities::route('/{record}/activities'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
                'organization'
            ]);
    }
}
