<?php

namespace App\Filament\Office\Resources;

use Filament\Tables;
use App\Models\Office;
use Filament\Forms\Form;
use Filament\Tables\Table;
use Filament\Resources\Resource;
use Filament\Support\Enums\Alignment;
use Filament\Support\Enums\FontWeight;
use Filament\Tables\Columns\TextColumn;
use Illuminate\Database\Eloquent\Model;
use Filament\Tables\Columns\Layout\Grid;
use Filament\Tables\Columns\Layout\Stack;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use App\Filament\Office\Resources\OfficeResource\Pages;
use App\Filament\Office\Resources\OfficeResource\OfficeForms;

class OfficeResource extends Resource
{
    protected static ?string $model = Office::class;

    protected static ?string $slug = 'companies/offices';
    protected static ?string $navigationGroup = 'Companies';
    protected static ?string $navigationIcon = 'heroicon-o-building-office';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                OfficeForms::getAddForm()
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Grid::make([
                    'lg' => 2,
                    '2xl' => 5,
                ])
                ->schema([
                    TextColumn::make('name')
                        ->formatStateUsing(fn(?Model $record): string => $record->name . ' (' . $record->code . ')')
                        ->searchable()
                        ->sortable()
                        ->grow(false)
                        ->weight(FontWeight::Bold),
                    TextColumn::make('type')->badge()
                        ->alignment(Alignment::End),
                    TextColumn::make('address')
                        ->description(function(?Model $record):string {
                            $desc = '';
                            if ($record->city != null) {
                                $desc .= $record->city->name;
                            }
                            
                            if ($record->state != null) {
                                if ($record->city != null) {
                                    $desc .= ' ⋅ ';
                                }
                                $desc .= $record->state->name;
                            }
                            
                            if ($record->country != null) {
                                if ($record->state != null) {
                                    $desc .= ' ⋅ ';
                                }
                                if ($record->city != null && $record->state == null) {
                                    $desc .= ' ⋅ ';
                                }
                                $desc .= $record->country->name;
                            }
                            return $desc;
                        })
                        ->formatStateUsing(fn (?Model $record): string => $record->address . (($record->post_code != null)?' ⋅ ' . $record->post_code:''))
                        ->searchable()
                        ->columnSpan('full'),
                    Stack::make([
                        TextColumn::make('mobile')
                            ->icon('heroicon-o-device-phone-mobile'),
                        TextColumn::make('tel')
                            ->icon('heroicon-o-phone'),
                        TextColumn::make('primary_email')
                            ->icon('heroicon-o-envelope'),
                        TextColumn::make('website')
                            ->icon('heroicon-o-globe-alt'),
                    ])->columnSpan([
                        'lg' => 'full',
                        '2xl' => 2,
                    ])
                    ->collapsible(),
                ]),
            ])
            ->filters([
                Tables\Filters\TrashedFilter::make(),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
                Tables\Actions\ForceDeleteAction::make(),
                Tables\Actions\RestoreAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                    Tables\Actions\RestoreBulkAction::make(),
                ]),
            ])
            ->contentGrid([
                'md' => 2,
                'xl' => 3,
            ])
            ->paginated([15, 25, 50, 100, 'all']);
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ManageOffices::route('/'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }
}
