<?php

namespace App\Filament\Office\Resources\OfficeResource;

use App\Enums\OfficeTypes;
use App\Models\OfficeUser;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\RichEditor;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;

final class OfficeForms
{
    static public function getAddForm() : Grid {
        return Grid::make('Add New Office')->schema([
            FileUpload::make('logo')
                ->imagePreviewHeight('250')
                // ->loadingIndicatorPosition('left')
                // ->panelAspectRatio('2:1')
                // ->panelLayout('integrated')
                // ->removeUploadedFileButtonPosition('right')
                // ->uploadButtonPosition('left')
                // ->uploadProgressIndicatorPosition('left')
                ->directory(fn () => 'office/' . date('Y') . '/' . date('m'))
                ->getUploadedFileNameForStorageUsing(
                    fn (TemporaryUploadedFile $file): string => (string) str($file->getClientOriginalName())
                        ->prepend(auth()->id().'-'),
                ),

            FileUpload::make('dark_logo')
                ->imagePreviewHeight('250')
                // ->loadingIndicatorPosition('left')
                // ->panelAspectRatio('2:1')
                // ->panelLayout('integrated')
                // ->removeUploadedFileButtonPosition('right')
                // ->uploadButtonPosition('left')
                // ->uploadProgressIndicatorPosition('left')
                ->directory(fn () => 'office/' . date('Y') . '/' . date('m'))
                ->getUploadedFileNameForStorageUsing(
                    fn (TemporaryUploadedFile $file): string => (string) str($file->getClientOriginalName())
                        ->prepend(auth()->id().'-'),
                ),

            TextInput::make('name')
                ->required()
                ->maxLength(100),
            TextInput::make('code')
                ->required()
                ->unique(ignoreRecord: true)
                ->maxLength(8)
                ->minLength(3),
            Textarea::make('address')
                ->required()
                ->maxLength(200)
                ->columnSpan(2),
            TextInput::make('mobile')->label('Mobile Number')
                ->maxLength(100),
            Select::make('type')
                ->options(OfficeTypes::class),
            Select::make('country_id')->label('Country')
                ->options(\App\Models\Country::all()->pluck('name', 'id'))
                ->reactive()
                ->afterStateUpdated(function (callable $set) {
                    $set('city_id', '');
                    $set('state_id', '');
                })
                ->searchable(),
            Select::make('city_id')->label('City')
                ->options(function(callable $get) {
                    $country = \App\Models\Country::find($get('country_id'));
                    if ($country) {
                        return $country->cities->pluck('name', 'id');
                    }
                    return [''=>'Please state/country first'];
                })
                ->searchable(),
            Select::make('state_id')->label('State')
                ->options(
                    function(callable $get) {
                        $country = \App\Models\Country::find($get('country_id'));
                        if ($country) {
                            return $country->states->pluck('name', 'id');
                        }
                        return [''=>'Please country first'];
                    }
                )
                ->reactive()
                ->searchable(),
            TextInput::make('post_code')
                    ->maxLength(20),
            Select::make('comapany_admin')
                ->options(\App\Models\OfficeUser::all()->pluck('name', 'id'))
                ->searchable(),
            TextInput::make('website')
                ->maxLength(100),
            TextInput::make('primary_email')
                ->email()
                ->maxLength(100),
            TextInput::make('tel')->label('Telephone Number')
                ->tel()
                ->maxLength(100),
            RichEditor::make('note')
                ->maxLength(65535)
                ->columnSpan(2),
            Repeater::make('office_staff_users')
                ->relationship()
                ->schema([
                    Select::make('office_user_id')
                        ->options(OfficeUser::all()->pluck('name', 'id')),
                ])
                ->columns(3)
        ])->columns(2);
    }
}