<?php

namespace App\Filament\Office\Resources;

use Filament\Forms;
use Filament\Tables;
use App\Models\Company;
use App\Models\Tracking;
use Filament\Forms\Form;
use Filament\Tables\Table;
use Illuminate\Support\Str;
use Filament\Resources\Resource;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Group;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\RichEditor;
use Filament\Tables\Filters\SelectFilter;
use Illuminate\Database\Eloquent\Builder;
use App\Filament\Office\Resources\TrackingResource\Pages;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use App\Filament\Office\Resources\TrackingResource\RelationManagers;

class TrackingResource extends Resource
{
    protected static ?string $model = Tracking::class;

    protected static ?string $navigationGroup = 'Shipping';
    protected static ?string $navigationIcon = 'heroicon-o-signal';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Group::make()
                    ->schema([
                        Section::make('Status')
                            ->schema([
                                TextInput::make('status_details')
                                    ->maxLength(255)
                                    ->columnSpanFull(),
                                RichEditor::make('description')
                                    ->columnSpanFull(),
                            ])->columns(['lg' => 2]),
                    ])
                    ->columnSpan(['lg' => 2]),
                Group::make()
                    ->schema([
                        Section::make('Relation')
                            ->schema([
                                Select::make('bl_id')->label('BL Number')
                                    ->relationship('bl', 'bl_no', fn (Builder $query) => $query->withTrashed())
                                    ->required()
                                    ->preload()
                                    ->searchable(),
                                DatePicker::make('date')
                                    ->required()
                                    ->default(today()),
                                Select::make('status_id')
                                    ->label('Status')
                                    ->options(\App\Models\Status::all()->pluck('status', 'id'))
                                    ->searchable()
                                    ->preload()
                                    ->required(),
                            ]),
                    ])
                    ->columnSpan(['lg' => 1]),
            ])
            ->columns(3);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('bl.bl_no')
                    ->label('BL Number')
                    ->description(fn (Tracking $record): string => \Carbon\Carbon::parse($record->date)->toFormattedDateString())
                    ->sortable()
                    ->searchable(),
                // Tables\Columns\TextColumn::make('port.name')
                //     ->numeric()
                //     ->sortable(),
                // Tables\Columns\TextColumn::make('vessel.name')
                //     ->numeric()
                //     ->sortable(),
                // Tables\Columns\TextColumn::make('date')
                //     ->date()
                //     ->since()
                //     ->sortable(),
                Tables\Columns\TextColumn::make('status.status')
                    ->description(fn (Tracking $record): string => ($record->port != null)?$record->port->name . (($record->vessel != null)?' ⋅ ' . $record->vessel->name:''):'')
                    ->formatStateUsing(fn ($state): string => Str::headline($state))
                    ->badge()
                    // ->color()
                    ->html()
                    ->searchable(),
                Tables\Columns\TextColumn::make('status_details')
                    ->searchable()
                    ->wrap()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('created_at')->label('Added')
                    ->dateTime()
                    ->since()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')->label('Updated')
                    ->dateTime()
                    ->since()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('deleted_at')->label('Deleted')
                    ->dateTime()
                    ->since()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->defaultSort('created_at', 'desc')
            ->filters([
                SelectFilter::make('status')
                    ->options(\App\Models\Status::all()->pluck('status', 'id')),
                    SelectFilter::make('port')
                        ->relationship('port', 'name')
                        ->searchable(),
                    SelectFilter::make('vessel')
                        ->relationship('vessel', 'name')
                        ->searchable(),
                Tables\Filters\TrashedFilter::make(),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                    Tables\Actions\RestoreBulkAction::make(),
                ]),
            ])
            ->emptyStateActions([
                Tables\Actions\CreateAction::make(),
            ]);
    }

    static protected function getPortAddForm() : Grid {
        return Grid::make('Add New Port')->schema([
            TextInput::make('name')
                ->helperText('Enter full name of port here (no iso or country code).')
                ->required()
                ->minLength(2)
                ->maxLength(100),
            Select::make('country_id')
                ->relationship('country', 'name')
                ->searchable()
                ->required(),
            TextInput::make('latitude')->numeric(),
            TextInput::make('longitude')->numeric(),
            TextInput::make('iso')->label('ISO')
                ->helperText('ISO is the unique identifier for a port.')
                ->minLength(2)
                ->maxLength(3)
                ->required(),
            Select::make('type')
                ->options([
                    'inland' => 'Inland',
                    'dry' => 'Dry',
                    'fishing' => 'Fishing',
                    'warm-water' => 'Warm Water',
                    'seaports' => 'Seaports',
                ]),
        ])->columns(2);
    }

    static protected function getVesselAddForm() : Grid {
        return Grid::make('Add New Delivery Agent')->schema([
            TextInput::make('name')
                ->unique(ignorable: fn ($record) => $record)
                ->helperText('Write new vessel name here')
                ->minLength(3)
                ->maxLength(100),
            TextInput::make('capacity')
                ->helperText('Capacity with tues')
                ->minLength(1),
            Select::make('route_start')
                ->label('Starting Port')
                ->relationship('port_start', 'name')
                ->searchable(),
            Select::make('route_end')
                ->label('Destination Port')
                ->relationship('port_end', 'name')
                ->searchable(),
        ])->columns(2);
    }
    
    public static function getRelations(): array
    {
        return [
            //
        ];
    }
    
    public static function getPages(): array
    {
        return [
            'index' => Pages\ListTrackings::route('/'),
            'create' => Pages\CreateTracking::route('/create'),
            'edit' => Pages\EditTracking::route('/{record}/edit'),
        ];
    }    
    
    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]);
    }
}
